export function buildToolMetadata(tool) {
    const methodName = toProxyMethodName(tool.name);
    const properties = extractOptions(tool);
    return {
        tool,
        methodName,
        options: properties,
    };
}
export function buildEmbeddedSchemaMap(tools) {
    const result = {};
    for (const entry of tools) {
        if (entry.tool.inputSchema && typeof entry.tool.inputSchema === 'object') {
            result[entry.tool.name] = entry.tool.inputSchema;
        }
    }
    return result;
}
export function extractOptions(tool) {
    const schema = tool.inputSchema;
    if (!schema || typeof schema !== 'object') {
        return [];
    }
    const record = schema;
    if (record.type !== 'object' || typeof record.properties !== 'object') {
        return [];
    }
    // Flatten schema properties into Commander-friendly option descriptors.
    const properties = record.properties;
    const requiredList = Array.isArray(record.required) ? record.required : [];
    return Object.entries(properties).map(([property, descriptor]) => {
        const type = inferType(descriptor);
        const enumValues = getEnumValues(descriptor);
        const defaultValue = getDescriptorDefault(descriptor);
        const formatInfo = getDescriptorFormatHint(descriptor);
        const placeholder = buildPlaceholder(property, type, enumValues, formatInfo?.slug);
        const exampleValue = buildExampleValue(property, type, enumValues, defaultValue);
        return {
            property,
            cliName: toCliOption(property),
            description: getDescriptorDescription(descriptor),
            required: requiredList.includes(property),
            type,
            placeholder,
            exampleValue,
            enumValues,
            defaultValue,
            formatHint: formatInfo?.display,
        };
    });
}
export function getEnumValues(descriptor) {
    if (!descriptor || typeof descriptor !== 'object') {
        return undefined;
    }
    const record = descriptor;
    if (Array.isArray(record.enum)) {
        const values = record.enum.filter((entry) => typeof entry === 'string');
        return values.length > 0 ? values : undefined;
    }
    if (record.type === 'array' && typeof record.items === 'object' && record.items !== null) {
        const nested = record.items;
        if (Array.isArray(nested.enum)) {
            const values = nested.enum.filter((entry) => typeof entry === 'string');
            return values.length > 0 ? values : undefined;
        }
    }
    return undefined;
}
export function getDescriptorDefault(descriptor) {
    if (!descriptor || typeof descriptor !== 'object') {
        return undefined;
    }
    const record = descriptor;
    if (record.default !== undefined) {
        return record.default;
    }
    if (record.type === 'array' && typeof record.items === 'object' && record.items !== null) {
        return Array.isArray(record.default) ? record.default : undefined;
    }
    return undefined;
}
export function buildPlaceholder(property, type, enumValues, formatSlug) {
    const normalized = property.replace(/[A-Z]/g, (char) => `-${char.toLowerCase()}`).replace(/_/g, '-');
    if (enumValues && enumValues.length > 0) {
        return `<${normalized}:${enumValues.join('|')}>`;
    }
    switch (type) {
        case 'number':
            return `<${normalized}:number>`;
        case 'boolean':
            return `<${normalized}:true|false>`;
        case 'array':
            return `<${normalized}:value1,value2>`;
        default:
            if (formatSlug) {
                return `<${normalized}:${formatSlug}>`;
            }
            return `<${normalized ?? 'value'}>`;
    }
}
export function buildExampleValue(property, type, enumValues, defaultValue) {
    if (enumValues && enumValues.length > 0) {
        return enumValues[0];
    }
    if (defaultValue !== undefined) {
        try {
            return typeof defaultValue === 'string' ? defaultValue : JSON.stringify(defaultValue);
        }
        catch {
            return undefined;
        }
    }
    switch (type) {
        case 'number':
            return '1';
        case 'boolean':
            return 'true';
        case 'array':
            return 'value1,value2';
        default:
            if (property.toLowerCase().includes('path')) {
                return '/path/to/file.md';
            }
            if (property.toLowerCase().includes('id')) {
                return 'example-id';
            }
            return undefined;
    }
}
export function pickExampleLiteral(option) {
    if (option.enumValues && option.enumValues.length > 0) {
        return JSON.stringify(option.enumValues[0]);
    }
    if (!option.exampleValue) {
        return undefined;
    }
    if (option.type === 'array') {
        const values = option.exampleValue
            .split(',')
            .map((entry) => entry.trim())
            .filter(Boolean);
        if (values.length === 0) {
            return undefined;
        }
        return `[${values.map((entry) => JSON.stringify(entry)).join(', ')}]`;
    }
    if (option.type === 'number' || option.type === 'boolean') {
        return option.exampleValue;
    }
    try {
        const parsed = JSON.parse(option.exampleValue);
        if (typeof parsed === 'number' || typeof parsed === 'boolean') {
            return option.exampleValue;
        }
    }
    catch {
        // fall through to quoted string literal
    }
    return JSON.stringify(option.exampleValue);
}
export function buildFallbackLiteral(option) {
    switch (option.type) {
        case 'number':
            return '1';
        case 'boolean':
            return 'true';
        case 'array':
            return '["value1"]';
        default: {
            if (option.property.toLowerCase().includes('id')) {
                return JSON.stringify('example-id');
            }
            if (option.property.toLowerCase().includes('url')) {
                return JSON.stringify('https://example.com');
            }
            return JSON.stringify('value');
        }
    }
}
export function inferType(descriptor) {
    if (!descriptor || typeof descriptor !== 'object') {
        return 'unknown';
    }
    const type = descriptor.type;
    if (type === 'string' || type === 'number' || type === 'boolean' || type === 'array') {
        return type;
    }
    return 'unknown';
}
export function getDescriptorDescription(descriptor) {
    if (typeof descriptor !== 'object' || descriptor === null) {
        return undefined;
    }
    const record = descriptor;
    return typeof record.description === 'string' ? record.description : undefined;
}
export function getDescriptorFormatHint(descriptor) {
    if (typeof descriptor !== 'object' || descriptor === null) {
        return undefined;
    }
    const record = descriptor;
    const formatRaw = typeof record.format === 'string' ? record.format : undefined;
    const description = typeof record.description === 'string' ? record.description : undefined;
    const iso8601FromDescription = !formatRaw && description && /\biso[-\s]*8601\b/i.test(description) ? 'iso-8601' : undefined;
    const isoFormatFromDescription = !formatRaw && !iso8601FromDescription && description && /\biso\s+format\b/i.test(description)
        ? 'iso-8601'
        : undefined;
    const formatFromDescription = iso8601FromDescription ?? isoFormatFromDescription;
    const slug = formatRaw ?? formatFromDescription;
    if (!slug) {
        return undefined;
    }
    let display;
    switch (slug) {
        case 'date-time':
        case 'iso-8601':
            display = 'ISO 8601';
            break;
        case 'uuid':
            display = 'UUID';
            break;
        default:
            display = slug.replace(/[_-]/g, ' ');
            display = display.charAt(0).toUpperCase() + display.slice(1);
            break;
    }
    return {
        display: display.replace(/\b\w/g, (char) => char.toUpperCase()),
        slug,
    };
}
export function toProxyMethodName(toolName) {
    return toolName
        .replace(/[-_](\w)/g, (_, char) => char.toUpperCase())
        .replace(/^(\w)/, (match) => match.toLowerCase());
}
export function toCliOption(property) {
    return property.replace(/[A-Z]/g, (char) => `-${char.toLowerCase()}`).replace(/_/g, '-');
}
export const toolsTestHelpers = {
    getEnumValues,
    getDescriptorDefault,
    buildPlaceholder,
    buildExampleValue,
    pickExampleLiteral,
    buildFallbackLiteral,
};
//# sourceMappingURL=tools.js.map