import { spawn } from 'node:child_process';
import { randomUUID } from 'node:crypto';
import http from 'node:http';
import { URL } from 'node:url';
import { buildOAuthPersistence } from './oauth-persistence.js';
const CALLBACK_HOST = '127.0.0.1';
const CALLBACK_PATH = '/callback';
// createDeferred produces a minimal promise wrapper for async coordination.
function createDeferred() {
    let resolve;
    let reject;
    const promise = new Promise((res, rej) => {
        resolve = res;
        reject = rej;
    });
    return { promise, resolve, reject };
}
// openExternal attempts to launch the system browser cross-platform.
function openExternal(url) {
    const platform = process.platform;
    const stdio = 'ignore';
    try {
        if (platform === 'darwin') {
            const child = spawn('open', [url], { stdio, detached: true });
            child.unref();
        }
        else if (platform === 'win32') {
            const child = spawn('cmd', ['/c', 'start', '""', url], {
                stdio,
                detached: true,
            });
            child.unref();
        }
        else {
            const child = spawn('xdg-open', [url], { stdio, detached: true });
            child.unref();
        }
    }
    catch {
        // best-effort: fall back to printing URL
    }
}
// PersistentOAuthClientProvider persists OAuth session artifacts to disk and captures callback redirects.
class PersistentOAuthClientProvider {
    definition;
    metadata;
    logger;
    persistence;
    redirectUrlValue;
    authorizationDeferred = null;
    server;
    constructor(definition, persistence, redirectUrl, logger) {
        this.definition = definition;
        this.redirectUrlValue = redirectUrl;
        this.logger = logger;
        this.persistence = persistence;
        this.metadata = {
            client_name: definition.clientName ?? `mcporter (${definition.name})`,
            redirect_uris: [this.redirectUrlValue.toString()],
            grant_types: ['authorization_code', 'refresh_token'],
            response_types: ['code'],
            token_endpoint_auth_method: 'none',
            scope: 'mcp:tools',
        };
    }
    static async create(definition, logger) {
        const persistence = await buildOAuthPersistence(definition, logger);
        const server = http.createServer();
        const overrideRedirect = definition.oauthRedirectUrl ? new URL(definition.oauthRedirectUrl) : null;
        const listenHost = overrideRedirect?.hostname ?? CALLBACK_HOST;
        const overridePort = overrideRedirect?.port ?? '';
        const usesDynamicPort = !overrideRedirect || overridePort === '' || overridePort === '0';
        const desiredPort = usesDynamicPort ? undefined : Number.parseInt(overridePort, 10);
        const callbackPath = overrideRedirect?.pathname && overrideRedirect.pathname !== '/' ? overrideRedirect.pathname : CALLBACK_PATH;
        const port = await new Promise((resolve, reject) => {
            server.listen(desiredPort ?? 0, listenHost, () => {
                const address = server.address();
                if (typeof address === 'object' && address && 'port' in address) {
                    resolve(address.port);
                }
                else {
                    reject(new Error('Failed to determine callback port'));
                }
            });
            server.once('error', (error) => reject(error));
        });
        const redirectUrl = overrideRedirect
            ? new URL(overrideRedirect.toString())
            : new URL(`http://${listenHost}:${port}${callbackPath}`);
        if (usesDynamicPort) {
            redirectUrl.port = String(port);
        }
        if (!overrideRedirect || overrideRedirect.pathname === '/' || overrideRedirect.pathname === '') {
            redirectUrl.pathname = callbackPath;
        }
        const provider = new PersistentOAuthClientProvider(definition, persistence, redirectUrl, logger);
        provider.attachServer(server);
        return {
            provider,
            close: async () => {
                await provider.close();
            },
        };
    }
    // attachServer listens for the OAuth redirect and resolves/rejects the deferred code promise.
    attachServer(server) {
        this.server = server;
        server.on('request', async (req, res) => {
            try {
                const url = req.url ?? '';
                const parsed = new URL(url, this.redirectUrlValue);
                const expectedPath = this.redirectUrlValue.pathname || '/callback';
                if (parsed.pathname !== expectedPath) {
                    res.statusCode = 404;
                    res.end('Not found');
                    return;
                }
                const code = parsed.searchParams.get('code');
                const error = parsed.searchParams.get('error');
                const receivedState = parsed.searchParams.get('state');
                const expectedState = await this.persistence.readState();
                if (expectedState && receivedState && receivedState !== expectedState) {
                    res.statusCode = 400;
                    res.setHeader('Content-Type', 'text/html');
                    res.end('<html><body><h1>Authorization failed</h1><p>Invalid OAuth state</p></body></html>');
                    this.authorizationDeferred?.reject(new Error('Invalid OAuth state'));
                    this.authorizationDeferred = null;
                    return;
                }
                if (code) {
                    this.logger.info(`Received OAuth authorization code for ${this.definition.name}`);
                    res.statusCode = 200;
                    res.setHeader('Content-Type', 'text/html');
                    res.end('<html><body><h1>Authorization successful</h1><p>You can return to the CLI.</p></body></html>');
                    this.authorizationDeferred?.resolve(code);
                    this.authorizationDeferred = null;
                }
                else if (error) {
                    res.statusCode = 400;
                    res.setHeader('Content-Type', 'text/html');
                    res.end(`<html><body><h1>Authorization failed</h1><p>${error}</p></body></html>`);
                    this.authorizationDeferred?.reject(new Error(`OAuth error: ${error}`));
                    this.authorizationDeferred = null;
                }
                else {
                    res.statusCode = 400;
                    res.end('Missing authorization code');
                    this.authorizationDeferred?.reject(new Error('Missing authorization code'));
                    this.authorizationDeferred = null;
                }
            }
            catch (error) {
                this.authorizationDeferred?.reject(error);
                this.authorizationDeferred = null;
            }
        });
    }
    get redirectUrl() {
        return this.redirectUrlValue;
    }
    get clientMetadata() {
        return this.metadata;
    }
    async state() {
        const existing = await this.persistence.readState();
        if (existing) {
            return existing;
        }
        const state = randomUUID();
        await this.persistence.saveState(state);
        return state;
    }
    async clientInformation() {
        return this.persistence.readClientInfo();
    }
    async saveClientInformation(clientInformation) {
        await this.persistence.saveClientInfo(clientInformation);
    }
    async tokens() {
        return this.persistence.readTokens();
    }
    async saveTokens(tokens) {
        await this.persistence.saveTokens(tokens);
        this.logger.info(`Saved OAuth tokens for ${this.definition.name} (${this.persistence.describe()})`);
    }
    async redirectToAuthorization(authorizationUrl) {
        this.logger.info(`Authorization required for ${this.definition.name}. Opening browser...`);
        this.authorizationDeferred = createDeferred();
        openExternal(authorizationUrl.toString());
        this.logger.info(`If the browser did not open, visit ${authorizationUrl.toString()} manually.`);
    }
    async saveCodeVerifier(codeVerifier) {
        await this.persistence.saveCodeVerifier(codeVerifier);
    }
    async codeVerifier() {
        const value = await this.persistence.readCodeVerifier();
        if (!value) {
            throw new Error(`Missing PKCE code verifier for ${this.definition.name}`);
        }
        return value.trim();
    }
    // invalidateCredentials removes cached files to force the next OAuth flow.
    async invalidateCredentials(scope) {
        await this.persistence.clear(scope);
    }
    // waitForAuthorizationCode resolves once the local callback server captures a redirect.
    async waitForAuthorizationCode() {
        if (!this.authorizationDeferred) {
            this.authorizationDeferred = createDeferred();
        }
        return this.authorizationDeferred.promise;
    }
    // close stops the temporary callback server created for the OAuth session.
    async close() {
        if (this.authorizationDeferred) {
            // If the CLI is tearing down mid-flow, reject the pending wait promise so runtime shutdown isn't blocked.
            this.authorizationDeferred.reject(new Error('OAuth session closed before receiving authorization code.'));
            this.authorizationDeferred = null;
        }
        if (!this.server) {
            return;
        }
        await new Promise((resolve) => {
            this.server?.close(() => resolve());
        });
        this.server = undefined;
    }
}
// createOAuthSession spins up a file-backed OAuth provider and callback server for the target definition.
export async function createOAuthSession(definition, logger) {
    const { provider, close } = await PersistentOAuthClientProvider.create(definition, logger);
    const waitForAuthorizationCode = () => provider.waitForAuthorizationCode();
    return {
        provider,
        waitForAuthorizationCode,
        close,
    };
}
//# sourceMappingURL=oauth.js.map