import { looksLikeHttpUrl, splitHttpToolSelector } from './http-utils.js';
import { chooseClosestIdentifier, renderIdentifierResolutionMessages } from './identifier-helpers.js';
import { dimText, yellowText } from './terminal.js';
const CALL_TOKEN_PATTERN = /[.(]/;
export function inferCommandRouting(token, args, definitions) {
    if (!token) {
        return { kind: 'command', command: token, args };
    }
    if (token === 'describe') {
        return { kind: 'command', command: 'list', args };
    }
    // Hidden alias kept for muscle memory / older docs.
    if (token === 'list-tools') {
        return { kind: 'command', command: 'list', args };
    }
    if (isExplicitCommand(token)) {
        return { kind: 'command', command: token, args };
    }
    if (isHttpToolToken(token)) {
        return { kind: 'command', command: 'call', args: [token, ...args] };
    }
    if (isUrlToken(token)) {
        return { kind: 'command', command: 'list', args: [token, ...args] };
    }
    if (isCallLikeToken(token)) {
        return { kind: 'command', command: 'call', args: [token, ...args] };
    }
    if (definitions.length === 0) {
        return { kind: 'command', command: token, args };
    }
    const serverNames = definitions.map((entry) => entry.name);
    if (serverNames.includes(token)) {
        return { kind: 'command', command: 'list', args: [token, ...args] };
    }
    const resolution = chooseClosestIdentifier(token, serverNames);
    if (!resolution) {
        return { kind: 'command', command: token, args };
    }
    const messages = renderIdentifierResolutionMessages({
        entity: 'server',
        attempted: token,
        resolution,
    });
    if (resolution.kind === 'auto' && messages.auto) {
        console.log(dimText(messages.auto));
        return { kind: 'command', command: 'list', args: [resolution.value, ...args] };
    }
    if (messages.suggest) {
        console.error(yellowText(messages.suggest));
    }
    console.error(`Unknown MCP server '${token}'.`);
    return { kind: 'abort', exitCode: 1 };
}
function isCallLikeToken(token) {
    if (!token) {
        return false;
    }
    if (looksLikeHttpUrl(token)) {
        return false;
    }
    return CALL_TOKEN_PATTERN.test(token);
}
function isExplicitCommand(token) {
    return token === 'list' || token === 'call' || token === 'auth';
}
function isUrlToken(token) {
    return looksLikeHttpUrl(token);
}
function isHttpToolToken(token) {
    if (!token) {
        return false;
    }
    return splitHttpToolSelector(token) !== null;
}
//# sourceMappingURL=command-inference.js.map