import fs from 'node:fs/promises';
import path from 'node:path';
import { bundleOutput, compileBundleWithBun, computeCompileTarget, resolveBundleTarget, } from './cli/generate/artifacts.js';
import { ensureInvocationDefaults, fetchTools, resolveServerDefinition } from './cli/generate/definition.js';
import { resolveRuntimeKind } from './cli/generate/runtime.js';
import { readPackageMetadata, writeTemplate } from './cli/generate/template.js';
import { buildToolMetadata, toolsTestHelpers } from './cli/generate/tools.js';
import { serializeDefinition } from './cli-metadata.js';
// generateCli produces a standalone CLI (and optional bundle/binary) for a given MCP server.
export async function generateCli(options) {
    const runtimeKind = await resolveRuntimeKind(options.runtime, options.compile);
    const bundlerKind = options.bundler ?? (runtimeKind === 'bun' ? 'bun' : 'rolldown');
    if (bundlerKind === 'bun' && runtimeKind !== 'bun') {
        throw new Error('--bundler bun currently requires --runtime bun.');
    }
    const timeoutMs = options.timeoutMs ?? 30_000;
    const { definition: baseDefinition, name } = await resolveServerDefinition(options.serverRef, options.configPath, options.rootDir);
    const { tools, derivedDescription } = await fetchTools(baseDefinition, name, options.configPath, options.rootDir);
    const definition = baseDefinition.description || !derivedDescription
        ? baseDefinition
        : { ...baseDefinition, description: derivedDescription };
    const toolMetadata = tools.map((tool) => buildToolMetadata(tool));
    const generator = await readPackageMetadata();
    const baseInvocation = ensureInvocationDefaults({
        serverRef: options.serverRef,
        configPath: options.configPath,
        rootDir: options.rootDir,
        runtime: runtimeKind,
        bundler: bundlerKind,
        outputPath: options.outputPath,
        bundle: options.bundle,
        compile: options.compile,
        timeoutMs,
        minify: options.minify ?? false,
    }, definition);
    const embeddedMetadata = {
        schemaVersion: 1,
        generatedAt: new Date().toISOString(),
        generator,
        server: {
            name,
            source: definition.source,
            definition: serializeDefinition(definition),
        },
        artifact: {
            path: '',
            kind: 'template',
        },
        invocation: baseInvocation,
    };
    let templateTmpDir;
    let templateOutputPath = options.outputPath;
    if (!templateOutputPath && options.compile) {
        const tmpPrefix = path.join(process.cwd(), 'tmp', 'mcporter-cli-');
        await fs.mkdir(path.dirname(tmpPrefix), { recursive: true });
        templateTmpDir = await fs.mkdtemp(tmpPrefix);
        templateOutputPath = path.join(templateTmpDir, `${name}.ts`);
    }
    const outputPath = await writeTemplate({
        outputPath: templateOutputPath,
        runtimeKind,
        timeoutMs,
        definition,
        serverName: name,
        tools: toolMetadata,
        generator,
        metadata: embeddedMetadata,
    });
    let bundlePath;
    let compilePath;
    try {
        const shouldBundle = Boolean(options.bundle ?? options.compile);
        if (shouldBundle) {
            const targetPath = resolveBundleTarget({
                bundle: options.bundle,
                compile: options.compile,
                outputPath,
            });
            bundlePath = await bundleOutput({
                sourcePath: outputPath,
                runtimeKind,
                targetPath,
                minify: options.minify ?? false,
                bundler: bundlerKind,
            });
            if (options.compile) {
                if (runtimeKind !== 'bun') {
                    throw new Error('--compile is only supported when --runtime bun');
                }
                const compileTarget = computeCompileTarget(options.compile, bundlePath, name);
                await compileBundleWithBun(bundlePath, compileTarget);
                compilePath = compileTarget;
                if (!options.bundle) {
                    await fs.rm(bundlePath).catch(() => { });
                    bundlePath = undefined;
                }
            }
        }
    }
    finally {
        if (templateTmpDir) {
            await fs.rm(templateTmpDir, { recursive: true, force: true }).catch(() => { });
        }
    }
    return { outputPath: options.outputPath ?? outputPath, bundlePath, compilePath };
}
export const __test = toolsTestHelpers;
//# sourceMappingURL=generate-cli.js.map